/**
 * 水淹分析
 * @param viewer 场景主视图
 * @param baseHeight 场景高度
 * @constructor FloodAnalysis()
 */
function FloodAnalysis(viewer, baseHeight) {
    let _self = this;
    _self.polygon = [];
    _self.polygonJW = [];
    _self.start = false;
    _self.pause = false;
    _self.speed = null;
    _self.viewer = viewer;
    _self.height = baseHeight + 1;
    _self.maxHeight = null;
    _self.baseHeight = baseHeight;
    _self.drawLayer = new Cesium.PrimitiveCollection();
    _self.drawHelper = new DrawGeometryHelper(viewer);
    _self.viewer.scene.primitives.add(_self.drawLayer);
    _self.material = new Cesium.Material({
        fabric: {
            type: 'Water',
            uniforms: {
                normalMap: 'Build/Cesium/Assets/Textures/waterNormals.jpg',
                frequency: 10.0,
                animationSpeed: 0.01,
                amplitude: 30
            }
        }
    });
    _self.raise = null;
    /**
     * 重置坐标点
     * @returns {Array}
     */
    function setPosition() {
        let points = [];
        for (let i = 0; i < _self.polygonJW.length; i++) {
            points.push(_self.polygonJW[i][0], _self.polygonJW[i][1], _self.height);
        }
        return points;
    }
    /**
     * 绘制水淹范围
     * @param name
     * @constructor
     */
    function DrawPolygon(name) {
        _self.drawHelper.startDrawingPolygon({
            callback: function (position, state) {
                _self.polygon.push(position);
                let point = cartesian2LonLat(position);
                _self.polygonJW.push(point);
                let height = point[2];
                if (!_self.maxHeight) {
                    _self.maxHeight = height
                } else {
                    _self.maxHeight = _self.maxHeight > height ? _self.maxHeight : height
                }
                if (state === "drawEnd") {
                    var drawprimitive = new Cesium.Primitive({
                        geometryInstances: new Cesium.GeometryInstance({
                            geometry: new Cesium.PolygonGeometry.fromPositions({
                                positions: Cesium.Cartesian3.fromDegreesArrayHeights(setPosition()),
                                perPositionHeight: true,
                                extrudedHeight: _self.baseHeight,
                            })
                        }),
                        asynchronous: false,
                        appearance: new Cesium.EllipsoidSurfaceAppearance({
                            aboveGround: false,
                            material: _self.material
                        }),
                    });
                    drawprimitive.update = function (context, frameState, commandList) {
                        drawprimitive = new Cesium.Primitive({
                            geometryInstances: new Cesium.GeometryInstance({
                                geometry: new Cesium.PolygonGeometry.fromPositions({
                                    positions: Cesium.Cartesian3.fromDegreesArrayHeights(setPosition()),
                                    perPositionHeight: true,
                                    extrudedHeight: _self.baseHeight,
                                })
                            }),
                            asynchronous: false,
                            appearance: new Cesium.EllipsoidSurfaceAppearance({
                                aboveGround: false,
                                material: _self.material
                            }),
                        });
                        drawprimitive.update(context, frameState, commandList);
                    };
                    console.log(_self.polygon);
                    _self.drawLayer.add(drawprimitive);
                    closeCSS(name);
                    setActiveCSS(true);
                }
            }
        })
    }
    /**
     * 关闭对应css
     * @param name
     */
    function closeCSS(name) {
        $(('#' + name + "-svg")).css('fill', '#ffffff');
        $('.flood').removeClass('flood');
    }
    /**
     * 控制对应css可被点击
     * @param active
     */
    function setActiveCSS(active) {
        if (active) {
            $('#flood-start').addClass('active')
            $('#flood-pause').addClass('active')
            $('#flood-stop').addClass('active')
        } else {
            $('#flood-start').removeClass('active')
            $('#flood-pause').removeClass('active')
            $('#flood-stop').removeClass('active')
        }
    }
    /**
     * 平面坐标转经纬度坐标
     * @param cartesian
     * @returns {*[]}
     */
    function cartesian2LonLat(cartesian) {
        var cartographic = Cesium.Cartographic.fromCartesian(cartesian);
        var lon = Cesium.Math.toDegrees(cartographic.longitude);
        var lat = Cesium.Math.toDegrees(cartographic.latitude);
        var height = cartographic.height;
        return [lon, lat, height];
    }
    /**
     * 绘制水淹范围
     * @param name
     * @constructor
     */
    FloodAnalysis.prototype.DrawPolygon = function (name) {
        DrawPolygon(name);
    };
    /**
     * 设置水淹速度
     * @param speed 水淹速度
     */
    FloodAnalysis.prototype.setFloodSpeed = function (speed) {
        _self.speed = speed;
    };
    /**
     * 模拟水淹效果
     * @param name
     */
    FloodAnalysis.prototype.startFlood = function (name) {
        _self.start = true;
        // _self.height = baseHeight + 1;
        if (!_self.pause) {
            _self.height = baseHeight + 1;
        } else {
            _self.pause = false;
        }
        _self.raise = setInterval(function () {
            if (_self.start) {
                if (_self.height <= _self.maxHeight) {
                    _self.height += _self.speed * 0.01;
                } else {
                    _self.stopFlood(name);
                }
            }
        }, 10);
    };
    /**
     * 暂停水淹
     */
    FloodAnalysis.prototype.pauseFlood = function () {
        _self.start = false;
        _self.pause = true;
    };
    /**
     * 停止水淹效果
     * @param name
     */
    FloodAnalysis.prototype.stopFlood = function (name) {
        closeCSS(name);
        _self.start = false;
        _self.pause = false;
        clearInterval(_self.raise);
    };
    /**
     * 清除水淹效果
     * @constructor
     */
    FloodAnalysis.prototype.Clean = function () {
        setActiveCSS(false);
        _self.start = false;
        _self.pause = false;
        _self.drawHelper.stopDrawing();
        _self.polygon = [];
        _self.polygonJW = [];
        _self.drawLayer.removeAll();
        _self.height = baseHeight + 1;
        clearInterval(_self.raise);
        _self.raise = null;
    }
}