/**
 * 三维飞行
 * @param viewer 场景主视图
 * @constructor
 */
function RoamAnalysis(viewer) {
    let _self = this;
    _self.polyline = [];
    _self.start = false;
    _self.stop = null;
    _self.target = null;
    _self.isStop = false;
    _self.isDraw = false;
    _self.viewer = viewer;
    _self.height = 1200;
    _self.repeatTime = 5;
    _self.repeatNum = 5;
    _self.drawLayer = new Cesium.PrimitiveCollection();
    _self.drawHelper = new DrawGeometryHelper(viewer);
    _self.viewer.scene.primitives.add(_self.drawLayer);

    /**
     * 绘制折线
     * @param name
     * @constructor
     */
    function DrawPolyLine(name) {
        _self.drawHelper.startDrawingPolyline({
            callback: function (position, state) {
                if (state === "drawClick") {
                    _self.polyline.push(cartesian2LonLat(position.clone()));
                } else if (state === "drawEnd") {
                    let property = setRoamRoad(_self.polyline);
                    addRoamRoad(property);
                    _self.drawHelper.stopDrawing();
                    closeCSS(name);
                }
            }
        });
    }

    /**
     * 加载默认飞行路径
     * @param polygon 飞行路线
     * @returns {Cesium.SampledPositionProperty} 飞行路径
     */
    function setRoamRoad(polygon) {
        _self.start = Cesium.JulianDate.fromDate(new Date());
        let time = _self.start.clone();
        let property = new Cesium.SampledPositionProperty();
        viewer.clock.startTime = _self.start.clone();
        viewer.clock.currentTime = _self.start.clone();
        viewer.clock.clockRange = Cesium.ClockRange.UNBOUNDED; //Loop at the end
        viewer.clock.multiplier = 1;
        viewer.clock.shouldAnimate = false;
        for (let i = 0; i < polygon.length - 1; i++) {
            let startPoint = polygon[i], endPoint = polygon[i + 1];
            for (let j = 0; j <= _self.repeatNum; j++) {
                time = Cesium.JulianDate.addSeconds(time, j * (_self.repeatTime / _self.repeatNum), new Cesium.JulianDate());
                let position = null;
                if (startPoint.length >= 2 && endPoint.length >= 2) {
                    position = Cesium.Cartesian3.fromDegrees(startPoint[0] + j * (endPoint[0] - startPoint[0]) / _self.repeatNum,
                        startPoint[1] + j * (endPoint[1] - startPoint[1]) / _self.repeatNum, startPoint[2] + j * (endPoint[2] - startPoint[2]) / _self.repeatNum,);
                } else {
                    continue;
                }
                property.addSample(time, position);
            }
        }
        _self.stop = time.clone();
        _self.viewer.clock.stopTime = _self.stop.clone();
        setActiveCSS(true);
        return property;
    }

    /**
     * 加载飞行路径
     * @param property
     */
    function addRoamRoad(property) {
        _self.target = _self.viewer.entities.add({
            availability: new Cesium.TimeIntervalCollection([new Cesium.TimeInterval({
                start: _self.start,
                stop: _self.stop
            })]),
            position: property,
            orientation: new Cesium.VelocityOrientationProperty(property),
            path: {
                resolution: 2,
                material: new Cesium.PolylineGlowMaterialProperty({
                    glowPower: 0.1,
                    color: Cesium.Color.YELLOW
                }),
                width: 10,
                show: false
            },
            point: {
                pixelSize: 0,
                color: Cesium.Color.TRANSPARENT,
                // show: false
            }
        });
    }

    /**
     * 平面坐标转经纬度
     * @param cartesian
     * @returns {*[]}
     */
    function cartesian2LonLat(cartesian) {
        var cartographic = Cesium.Cartographic.fromCartesian(cartesian);
        var lon = Cesium.Math.toDegrees(cartographic.longitude);
        var lat = Cesium.Math.toDegrees(cartographic.latitude);
        return [lon, lat];
    }

    /**
     * 关闭css
     * @param name
     */
    function closeCSS(name) {
        $(('#' + name + "-svg")).css('fill', '#ffffff');
        $('.roams').removeClass('roams');
    }

    /**
     * 控制按钮点击
     * @param active
     */
    function setActiveCSS(active) {
        if (active) {
            $('#roam-start').addClass('active');
            $('#roam-pause').addClass('active');
            $('#roam-stop').addClass('active')
        } else {
            $('#roam-start').removeClass('active');
            $('#roam-pause').removeClass('active');
            $('#roam-stop').removeClass('active')
        }
    }

    /**
     * 飞行
     * @param clock
     */
    var setview = function (clock) {
        if (Cesium.JulianDate.lessThanOrEquals(_self.viewer.clock.stopTime, clock._currentTime)) {
            _self.stopRoam();
        }
    };
    /**
     * 开始飞行
     */
    RoamAnalysis.prototype.startRoam = function () {
        if (_self.isStop) {
            _self.viewer.clock.currentTime = _self.viewer.clock.startTime;
            _self.isStop = !_self.isStop;
        }
        _self.viewer.clock.onTick.addEventListener(setview);
        _self.viewer.clock.shouldAnimate = true;
        _self.viewer.trackedEntity = _self.target;
    };
    /**
     * 暂停飞行
     */
    RoamAnalysis.prototype.pauseRoam = function () {
        _self.viewer.clock.shouldAnimate = false;
    };
    /**
     * 停止飞行
     */
    RoamAnalysis.prototype.stopRoam = function () {
        _self.isStop = true;
        _self.viewer.clock.shouldAnimate = false;
        _self.viewer.trackedEntity = null;
        _self.viewer.clock.onTick.removeEventListener(setview);
        closeCSS("roam-start");
    };
    /**
     * 加载默认路径
     */
    RoamAnalysis.prototype.loadDefaultRoamRoad = function () {
        _self.Clean();
        // let property = setRoamRoad([[108.95836697787028, 34.21802468408287], [108.95842342171491, 34.221000678126835], [108.96048362968126, 34.22100346102044], [108.96046838017541, 34.21795541744988], [108.95836697787028, 34.21802468408287]]);
        let property = setRoamRoad([[88.34792113920346, 43.276293482889216, 1046.5804990777203],
            [88.34361064795858, 43.28180225890288, 1108.4392688320504],
            [88.34196654985949, 43.28386582721194, 1101.5677867290417],
            [88.33838962858111, 43.28841283764549, 1118.5930097368487],
            [88.33287327504226, 43.29540390832142, 1195.193324790287],
            [88.33030078294227, 43.29868667834923, 1261.050414218196],
            [88.32594686291488, 43.30240898887591, 1270.888694598863],
            [88.32373969860538, 43.30558006747427, 1323.7109103086977],
            [88.32102348439491, 43.30947214966523, 1382.081473079826],
            [88.31898070763353, 43.3143334517121, 1595.4085932399007],
            [88.31531152321566, 43.316732084063496, 1634.7777250143063],
            [88.31011393982628, 43.31999317538331, 1433.1187427635546],
            [88.31011393982604, 43.31999317538539, 1433.1187425025917],
            [88.31011393982604, 43.31999317538539, 1433.1187425025917]
        ]);
        addRoamRoad(property);
    };
    /**
     * 绘制自定义路径
     * @param name
     */
    RoamAnalysis.prototype.drawRoamRoad = function (name) {
        _self.Clean();
        DrawPolyLine(name);
    };
    /**
     * 清除路径飞行
     * @constructor
     */
    RoamAnalysis.prototype.Clean = function () {
        _self.drawHelper.stopDrawing();
        _self.drawLayer.removeAll();
        _self.polyline = [];
        _self.start = false;
        _self.stop = null;
        _self.isStop = false;
        _self.isDraw = false;
        if (_self.target) {
            _self.viewer.entities.remove(_self.target);
            _self.target = null;
        }
        _self.stopRoam();
        setActiveCSS(false);

    };

}