/**
 * 剖面分析
 * @param viewer 场景主视图
 * @param echart echart表格视图
 * @constructor
 */
function ProfileAnalysis(viewer, echart) {
    let _self = this;
    _self.point = [];
    _self.polyline = [];
    _self.viewer = viewer;
    _self.echart = echart;
    _self.drawLayer = new Cesium.PrimitiveCollection();
    _self.drawHelper = new DrawGeometryHelper(viewer);
    _self.viewer.scene.primitives.add(_self.drawLayer);

    /**
     * 绘制切线
     * @param name
     * @constructor
     */
    function DrawPolyLine(name) {
        _self.drawHelper.startDrawingPolyline({
            callback: function (position, state) {
                if (state === "drawClick") {
                    _self.point.push(position.clone());
                    _self.polyline.push(position.clone());
                }
                if (_self.polyline.length === 2) {
                    _self.drawHelper.stopDrawing();
                    var primitive = new Cesium.GroundPrimitive({
                        geometryInstances: new Cesium.GeometryInstance({
                            geometry: new Cesium.CorridorGeometry({
                                vertexFormat: Cesium.VertexFormat.POSITION_ONLY,
                                positions: _self.polyline,
                                width: 1
                            }),
                            attributes: {
                                color: Cesium.ColorGeometryInstanceAttribute.fromColor(Cesium.Color.fromAlpha(Cesium.Color.RED, 1))
                            }
                        }),
                        appearance: new Cesium.PerInstanceColorAppearance({
                            translucent: true,
                            closed: true,
                            aboveGround: true
                        }),
                        classificationType: Cesium.ClassificationType.BOTH
                    });
                    _self.drawLayer.add(primitive);
                    let startP = _self.point[0], endP = _self.point[1];
                    let startWP = Cesium.SceneTransforms.wgs84ToWindowCoordinates(_self.viewer.scene, startP);
                    let endWP = Cesium.SceneTransforms.wgs84ToWindowCoordinates(_self.viewer.scene, endP);
                    let num = parseInt(Cesium.Cartesian3.distance(startP, endP));
                    setProfile(new Cesium.Cartesian2(startWP.x, startWP.y), new Cesium.Cartesian2(endWP.x, endWP.y), num);
                    closeCSS(name);
                }
            }
        });
    }

    /**
     * 剖面分析
     * @param start
     * @param end
     * @param num
     */
    function setProfile(start, end, num) {
        var lons = [], lats = [], height = [], xheight = [], starth = null, lasth = null;
        for (var i = 0; i < num; i++) {
            let lon = Cesium.Math.lerp(start.x, end.x, 1 / num * (i + 1));
            let lat = Cesium.Math.lerp(start.y, end.y, 1 / num * (i + 1));
            let p = viewer.scene.pickPosition(new Cesium.Cartesian2(lon, lat));
            let h = (cartesian2LonLat(p)[2]).toFixed(1);
            let xh = (h - starth).toFixed(1);
            if (lasth) {
                if (lasth !== h) {
                    lons.push(lon);
                    lats.push(lat);
                    height.push(h);
                    xheight.push(xh);
                }
            } else {
                starth = h;
            }
            lasth = h;
        }
        setEchart(xheight);
    }

    /**
     * 在echart中显示分析结果
     * @param data
     */
    function setEchart(data) {
        let option = {
            xAxis: {
                data: []
            },
            yAxis: {},
            series: [{
                name: '切面相对高度',
                type: 'line',
                data: data,
                smooth: true
            }]
        };
        _self.echart.setOption(option);
        $("#echart-profile").show();
    }

    /**
     * 平面坐标转经纬度
     * @param cartesian
     * @returns {*[]}
     */
    function cartesian2LonLat(cartesian) {
        var cartographic = Cesium.Cartographic.fromCartesian(cartesian);
        var lon = Cesium.Math.toDegrees(cartographic.longitude);
        var lat = Cesium.Math.toDegrees(cartographic.latitude);
        return [lon, lat, cartographic.height];
    }

    /**
     * 关闭css
     * @param name
     */
    function closeCSS(name) {
        $(('#' + name + "-svg")).css('fill', '#ffffff');
        $('.profile').removeClass('profile');
    }

    /**
     * 绘制切线
     * @param name
     */
    ProfileAnalysis.prototype.drawPolyline = function (name) {
        DrawPolyLine(name);
    };
    /**
     * 清除剖面分析结果
     */
    ProfileAnalysis.prototype.clean = function () {
        $("#echart-profile").hide();
        _self.drawHelper.stopDrawing();
        _self.point = [];
        _self.polyline = [];
        _self.drawLayer.removeAll();
    };

}