import Vue from 'vue';
import axios from 'axios';
import qs from 'qs';
import './loginInterceptor';
import * as HTTP from '@config/http.config';

// 携带cookie信息
axios.defaults.withCredentials = true;
// 设置全局的请求次数，请求的间隙
axios.defaults.retry = 0;
axios.defaults.retryDelay = 1000;

// 请求超时拦截，重新请求
axios.interceptors.response.use(undefined, function axiosRetryInterceptor(err) {
  const config = err.config;
  // If config does not exist or the retry option is not set, reject
  if (!config || !config.retry) return Promise.reject(err);

  // Set the variable for keeping track of the retry count
  config.__retryCount = config.__retryCount || 0;

  // Check if we've maxed out the total number of retries
  if (config.__retryCount >= config.retry) {
    // Reject with the error
    return Promise.reject(err);
  }

  // Increase the retry count
  config.__retryCount += 1;

  // Create new promise to handle exponential backoff
  const backoff = new Promise(function(resolve) {
    setTimeout(function() {
      resolve();
    }, config.retryDelay || 1);
  });

  // Return the promise in which recalls axios to retry the request
  return backoff.then(function() {
    return axios(config);
  });
});

// 为Vue注册$http
Vue.prototype.$http = axios;

const CancelToken = axios.CancelToken;
let cancel;

const baseUrl = HTTP.baseUrl;
window.baseUrl = baseUrl;

const config = {
  baseUrl,

  /**
   * get获取数据，通用方法
   * @param {String} url
   * @param {Object} params
   * @param {Object} options
   */
  doGetPromise(url, params, options = {}) {
    const { timeout = 30000, ...arg } = options;

    return new Promise((resolve, reject) => {
      axios
        .get(url, {
          timeout: timeout,
          ...arg,
          params: {
            ...params,
            t: new Date().getTime() // 解决IE上get请求缓存问题
          },
          cancelToken: new CancelToken(function executor(c) {
            cancel = c;
          })
        })
        .then(response => {
          resolve(response.data);
        })
        .catch(response => {
          console.error('ajax error:', response);
          reject(response);
        });
    });
  },

  /**
   * FormData数据上传，文件上传必用
   * @param {String} url
   * @param {FormData} formData
   */
  doPostPromiseForm(url, formData) {
    return new Promise((resolve, reject) => {
      axios
        .post(url, formData, {
          headers: {
            'Content-type': 'multipart/form-data'
          },
          emulateJSON: false,
          emulateHTTP: false,
          cancelToken: new CancelToken(function executor(c) {
            cancel = c;
          })
        })
        .then(res => {
          resolve(res.data);
        })
        .catch(res => {
          console.log('post error:', res);
          reject(res);
        });
    });
  },

  /**
   * 默认方式提交from表单数据json
   * @param {String} url
   * @param {Object} data
   */
  doPostPromise(url, data) {
    return new Promise((resolve, reject) => {
      axios
        .post(url, qs.stringify(data), {
          headers: {
            'Content-type': 'application/x-www-form-urlencoded'
          },
          cancelToken: new CancelToken(function executor(c) {
            cancel = c;
          })
        })
        .then(res => {
          resolve(res.data);
        })
        .catch(res => {
          console.log('post error:', res);
          reject(res);
        });
    });
  },

  /**
   * 提交application/json方式
   * @param {String} url
   * @param {Object} data
   */
  doPostPromiseJson(url, data) {
    return new Promise((resolve, reject) => {
      axios
        .post(url, data, {
          headers: {
            'Content-type': 'application/json'
          },
          cancelToken: new CancelToken(function executor(c) {
            cancel = c;
          })
        })
        .then(res => {
          resolve(res.data);
        })
        .catch(res => {
          console.log('post error:', res);
          reject(res);
        });
    });
  }
};

// 切换页面强行中断请求 router.beforeEach中用到
Vue.prototype.$cancelAjax = function(msg) {
  if (cancel) {
    cancel(msg || '手动中断请求');
  }
};

export default config;
