import Vue from 'vue';
import axios from 'axios';
import qs from 'qs';
import store from '../store';
import * as _config from '@config/http.config';
import { Message } from 'element-ui';

// 携带cookie信息
axios.defaults.withCredentials = true;
// 设置全局的请求次数，请求的间隙
axios.defaults.retry = 1;
axios.defaults.retryDelay = 1000;
// 设置defaults.baseURL
axios.defaults.baseURL = _config.baseUrl;

// 请求超时拦截，重新请求
axios.interceptors.response.use(function(response) {
  const status = response.status;
  // const data = response.data;

  if (status !== 200) {
    Message({
      type: 'error',
      message: '请求服务好像出错了，' + status
    });
    return Promise.reject(response);
  }/*  else if (data.status === 2) {
    Message({
      type: 'error',
      message: '请求服务后台处理出现了错误'
    });
  } else if (data.status === 1) {
    Message({
      type: 'warning',
      message: '请求服务处理出现了异常，' + data.message
    });
  } */

  return Promise.resolve(response);
}, function axiosRetryInterceptor(err) {
  const config = err.config;
  // If config does not exist or the retry option is not set, reject
  if (!config || !config.retry) return Promise.reject(err);

  // Set the variable for keeping track of the retry count
  config.__retryCount = config.__retryCount || 0;

  // Check if we've maxed out the total number of retries
  if (config.__retryCount >= config.retry) {
    const status = err.response.status;
    Message({
      type: 'error',
      message: `请求服务好像出错了，状态码：${status}`
    });
    // Reject with the error
    return Promise.reject(err);
  }

  // Increase the retry count
  config.__retryCount += 1;

  // Create new promise to handle exponential backoff
  const backoff = new Promise(function(resolve) {
    setTimeout(function() {
      resolve();
    }, config.retryDelay || 1);
  });

  // Return the promise in which recalls axios to retry the request
  return backoff.then(function() {
    return axios(config);
  });
});

const CancelToken = axios.CancelToken;
let cancel;

const config = {
  /**
   * get获取数据，通用方法
   * @param {String} url
   * @param {Object} params
   * @param {Object} options
   */
  doGetPromise(url, params, options = {}) {
    const { timeout = 30000, ...arg } = options;

    return new Promise((resolve, reject) => {
      axios.get(url, {
        timeout: timeout,
        ...arg,
        params: {
          systemId: store.state.platformInfo.systemId, // 全面接口添加systemId字段
          ...params,
          t: new Date().getTime() // 解决IE上get请求缓存问题
        },
        cancelToken: new CancelToken(function executor(c) {
          cancel = c;
        })
      }).then(response => {
        resolve(response.data);
      }).catch(response => {
        // console.error('ajax error:', response)
        reject(response);
      });
    });
  },

  /**
   * FormData数据上传，文件上传必用
   * @param {String} url
   * @param {FormData} formData
   */
  doPostPromiseForm(url, formData) {
    return new Promise((resolve, reject) => {
      // 全面接口添加systemId字段
      if (formData.has) { // ie FormData没有has方法
        if (formData.has('systemId')) formData.append('systemId', store.state.platformInfo.systemId);
      }
      axios.post(url, formData, {
        headers: {
          'Content-type': 'multipart/form-data'
        },
        emulateJSON: false,
        emulateHTTP: false,
        cancelToken: new CancelToken(function executor(c) {
          cancel = c;
        })
      }).then(res => {
        resolve(res.data);
      }).catch(res => {
        console.log('post error:', res);
        reject(res);
      });
    });
  },

  /**
   * 默认方式提交from表单数据
   * @param {String} url
   * @param {Object} data
   */
  doPostPromise(url, data) {
    return new Promise((resolve, reject) => {
      // 全面接口添加systemId字段
      if (!data.hasOwnProperty('systemId')) data.systemId = store.state.platformInfo.systemId;
      axios.post(url, qs.stringify(data), {
        headers: {
          'Content-type': 'application/x-www-form-urlencoded'
        },
        cancelToken: new CancelToken(function executor(c) {
          cancel = c;
        })
      }).then(res => {
        resolve(res.data);
      }).catch(res => {
        // console.log('post error:', res)
        reject(res);
      });
    });
  },

  /**
   * 默认方式提交json数据
   * @param {String} url
   * @param {Object} data
   */
  doPostPromiseJson(url, data) {
    return new Promise((resolve, reject) => {
      // 全面接口添加systemId字段
      if (!data.hasOwnProperty('systemId')) data.systemId = store.state.platformInfo.systemId;
      axios.post(url, data, {
        headers: {
          'Content-type': 'application/json'
        },
        cancelToken: new CancelToken(function executor(c) {
          cancel = c;
        })
      }).then(res => {
        resolve(res.data);
      }).catch(res => {
        console.log('post error:', res);
        reject(res);
      });
    });
  }
};

// 切换页面强行中断请求 router.beforeEach中用到
Vue.prototype.$cancelAjax = function(msg) {
  if (cancel) {
    cancel(msg || '手动中断请求');
  }
};

export default config;
