import store from '@/store';
import Utils from '@/utils';
import { checkAsyncComponents } from '@/plugins/asyncComponents';

/**
 * 菜单表达式函数
 * 操作标签页
 */
export default {
  /**
 * 全局打开新标签页方法
 * 利用的是store, id作为唯一性检查，没有id就会自动生成id
 * @param {Object} data 组件的对象，比如 {id:'', name:'组织构架管理', componentName:'frameworkManage'}
 * @param {Any} param 绑定到数据里的param
 */
  openTab(data, param) {
    if (!data.id) data.id = Utils.generateId();
    let componentName = data.componentName || data.router;

    // 判断是否为http开头的外链
    if (componentName.indexOf('http') === 0) {
      param = {
        url: componentName
      };
      componentName = 'appIframe';
    } else {
      // 判断是否异步注册该组件
      const isRegistered = checkAsyncComponents(componentName);

      if (!isRegistered) {
        componentName = 'errorPage';
      }
    }

    // 菜单模块ID赋值，权限通过模块ID获取的
    if (!data.moduleId) {
      const menuList = store.getters['menu/menuList'];

      if (menuList.length > 0) {
        let menu;
        const fn = (data) => {
          const len = data.length;
          for (let i = 0; i < len; i++) {
            if (data[i].router === componentName) {
              menu = data[i];
              break;
            } else {
              if (data[i].children && data[i].children.length > 0) {
                fn(data[i].children);
              }
            }
          }
        };
        fn(menuList);

        // 模块id赋值
        if (menu) data.moduleId = menu.moduleId;
      }
    }

    data.componentName = componentName;

    if (param) data.param = param;

    store.dispatch('setNavigation', data);
  },
  /**
   * 打开一个外部链接的标签
   * @param {String} name 标签名称
   * @param {String} url 外部URL
   * @param {Boolean|String} only 布尔值为true时，是固定使用externalLinkTab---1打开页面，当是string字符串时，自定义打开固定标签页
   */
  openExternalLinkTab(name, url, only) {
    const id = (typeof only === 'boolean' && only)
      ? 'externalLinkTab---1'
      : (typeof only === 'string' && only)
        ? only : '';
    this.openTab({
      id,
      name,
      componentName: url
    });
  },
  /**
 * 关闭/移除标签页
 * @param {String} id tab标签的唯一标识
 */
  removeTab(id) {
    window.removeTab(id);  // 用的是Main.vue里的tab标签移除方法
  },
  /**
 * 刷新标签页
 * @param {String} id tab标签的唯一标识
 */
  refreshTab(id) {
    store.state.navigationList.forEach(item => {
      if (item.id === id) {
        item.param = {
          reload: new Date()
        };
      }
    });
  },
  // 获取当前打开的标签页
  getCurrentTab() {
    return store.state.currentNavigation;
  }
};
